#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Animation interactive pour la formation sur les risques liés à l'investissement obligataire.
Ce script permet de visualiser les propriétés de la duration et de la convexité d'un titre à taux fixe.
"""

import numpy as np
import matplotlib.pyplot as plt
from matplotlib.widgets import Slider, Button
from matplotlib.gridspec import GridSpec
import mplcursors


# Définition des fonctions de calcul pour les obligations
def calculer_prix_obligation(coupon_rate, yield_rate, maturity, face_value=10000):
    """
    Calcule le prix d'une obligation à taux fixe.

    Args:
        coupon_rate (float): Taux de coupon annuel (en décimal)
        yield_rate (float): Rendement annuel (en décimal)
        maturity (int): Maturité en années
        face_value (float): Valeur nominale

    Returns:
        float: Prix de l'obligation
    """
    if yield_rate == 0:
        return face_value * (1 + coupon_rate * maturity)

    coupon_payment = face_value * coupon_rate
    price = 0

    # Calcul de la valeur actualisée des coupons
    for t in range(1, maturity + 1):
        price += coupon_payment / ((1 + yield_rate) ** t)

    # Ajout de la valeur actualisée du principal
    price += face_value / ((1 + yield_rate) ** maturity)

    return price


def calculer_duration_macaulay(coupon_rate, yield_rate, maturity, face_value=10000):
    """
    Calcule la duration de Macaulay d'une obligation.

    Args:
        coupon_rate (float): Taux de coupon annuel (en décimal)
        yield_rate (float): Rendement annuel (en décimal)
        maturity (int): Maturité en années
        face_value (float): Valeur nominale

    Returns:
        float: Duration de Macaulay
    """
    if yield_rate == 0:
        # Cas spécial pour éviter la division par zéro
        weighted_sum = 0
        total_pv = 0

        for t in range(1, maturity + 1):
            pv = face_value * coupon_rate
            weighted_sum += t * pv
            total_pv += pv

        weighted_sum += maturity * face_value
        total_pv += face_value

        return weighted_sum / total_pv

    price = calculer_prix_obligation(coupon_rate, yield_rate, maturity, face_value)
    coupon_payment = face_value * coupon_rate
    weighted_sum = 0

    # Calcul de la somme pondérée des flux
    for t in range(1, maturity + 1):
        pv = coupon_payment / ((1 + yield_rate) ** t)
        weighted_sum += t * pv

    # Ajout de la valeur actualisée pondérée du principal
    weighted_sum += maturity * (face_value / ((1 + yield_rate) ** maturity))

    return weighted_sum / price


def calculer_duration_modifiee(coupon_rate, yield_rate, maturity, face_value=10000):
    """
    Calcule la duration modifiée d'une obligation.

    Args:
        coupon_rate (float): Taux de coupon annuel (en décimal)
        yield_rate (float): Rendement annuel (en décimal)
        maturity (int): Maturité en années
        face_value (float): Valeur nominale

    Returns:
        float: Duration modifiée
    """
    macaulay_duration = calculer_duration_macaulay(coupon_rate, yield_rate, maturity, face_value)
    return macaulay_duration / (1 + yield_rate)


def calculer_convexite(coupon_rate, yield_rate, maturity, face_value=10000):
    """
    Calcule la convexité d'une obligation.

    Args:
        coupon_rate (float): Taux de coupon annuel (en décimal)
        yield_rate (float): Rendement annuel (en décimal)
        maturity (int): Maturité en années
        face_value (float): Valeur nominale

    Returns:
        float: Convexité
    """
    if yield_rate == 0:
        # Cas spécial pour éviter la division par zéro
        weighted_sum = 0
        total_pv = 0

        for t in range(1, maturity + 1):
            pv = face_value * coupon_rate
            weighted_sum += t * (t + 1) * pv
            total_pv += pv

        weighted_sum += maturity * (maturity + 1) * face_value
        total_pv += face_value

        return weighted_sum / (total_pv * (1 + yield_rate) ** 2)

    price = calculer_prix_obligation(coupon_rate, yield_rate, maturity, face_value)
    coupon_payment = face_value * coupon_rate
    weighted_sum = 0

    # Calcul de la somme pondérée des flux pour la convexité
    for t in range(1, maturity + 1):
        pv = coupon_payment / ((1 + yield_rate) ** t)
        weighted_sum += t * (t + 1) * pv

    # Ajout de la valeur actualisée pondérée du principal
    weighted_sum += maturity * (maturity + 1) * (face_value / ((1 + yield_rate) ** maturity))

    return weighted_sum / (price * (1 + yield_rate) ** 2)


class ObligationAnimation:
    def __init__(self):
        # Paramètres initiaux
        self.face_value = 10000  # Valeur nominale de l'obligation
        self.coupon_rate_init = 0.06  # Taux de coupon initial (5%)
        self.yield_rate_init = 0.0760  # Taux de rendement initial (5%)
        self.maturity_init = 7  # Durée initiale (10 ans)

        # Plages pour les graphiques
        self.yield_range = np.linspace(0.01, 0.15, 100)  # 1% à 10%
        self.maturity_range = np.array(range(2, 20))  # 1 à 30 ans

        # Création de la figure
        self.fig = plt.figure(figsize=(14, 9), dpi=100, constrained_layout=False)
        self.fig.suptitle('ANALYSE DES RISQUES OBLIGATAIRES: DURATION ET CONVEXITÉ', fontsize=18, fontweight='bold')

        # Utilisation de GridSpec pour organiser la mise en page
        # 8 lignes x 3 colonnes
        # Les 3 premières lignes pour la première rangée de graphiques
        # Les 3 suivantes pour la deuxième rangée (décalée vers le bas comme demandé)
        # Les 2 dernières pour les sliders (réduits en taille)
        gs = GridSpec(8, 3, figure=self.fig, height_ratios=[3, 3, 0.5, 3, 3, 0.15, 1, 1], hspace=1.2)

        plt.subplots_adjust(wspace=0.175, hspace=0.8, top=0.89, right=0.918, bottom=0.08, left=0.112)
        # Première rangée de graphiques (lignes 0-1, étendue sur 1 colonne chacun)
        self.ax_price_yield = self.fig.add_subplot(gs[0:2, 0])
        self.ax_duration_yield = self.fig.add_subplot(gs[0:2, 1])
        self.ax_convexity_yield = self.fig.add_subplot(gs[0:2, 2])

        # Deuxième rangée de graphiques (lignes 3-4, avec espacement vers le bas)
        self.ax_duration_maturity = self.fig.add_subplot(gs[3:5, 0])
        self.ax_convexity_maturity = self.fig.add_subplot(gs[3:5, 1])
        self.ax_table = self.fig.add_subplot(gs[3:5, 2])


        # Sliders (lignes 6-7, réduits en hauteur)
        self.ax_coupon_slider = self.fig.add_subplot(gs[6, :])
        self.ax_yield_slider = self.fig.add_subplot(gs[7, 0:1])  # Plus petit
        self.ax_maturity_slider = self.fig.add_subplot(gs[7, 1:])  # Plus petit

        # self.ax_coupon_slider.set_position([0.2,0.15,0.2,0.02])
        # self.ax_yield_slider = self.fig.add_subplot([0.15,0.15,0.7,0.03])  # Plus petit
        # self.ax_maturity_slider = self.fig.add_subplot([0.15,0.15,0.7,0.03])

        # Création des sliders
        self.coupon_slider = Slider(
            self.ax_coupon_slider, 'Taux Coupon (%)',
            0.0, 15.0, valinit=self.coupon_rate_init * 100, valstep=0.1,
            color='skyblue'
        )

        self.yield_slider = Slider(
            self.ax_yield_slider, 'Rendement (%)',
            0.0, 15.0, valinit=self.yield_rate_init * 100, valstep=0.1,
            color='lightgreen'
        )

        self.maturity_slider = Slider(
            self.ax_maturity_slider, 'Maturité (années)',
            2.0, 20, valinit=self.maturity_init, valstep=1,
            color='salmon'
        )

        # Connecter les callbacks
        self.coupon_slider.on_changed(self.update)
        self.yield_slider.on_changed(self.update)
        self.maturity_slider.on_changed(self.update)

        # Affichage initial
        self.update(None)

    def update(self, val):
        # Récupération des valeurs courantes
        coupon_rate = self.coupon_slider.val / 100
        yield_rate = self.yield_slider.val / 100
        maturity = int(self.maturity_slider.val)

        # Sauvegarder les valeurs précédentes pour effet de surbrillance si changement
        if hasattr(self, 'previous_values'):
            prev_values = self.previous_values
        else:
            # Valeurs initiales
            prev_values = {
                'coupon_rate': self.coupon_rate_init,
                'yield_rate': self.yield_rate_init,
                'maturity': self.maturity_init
            }

        # Détecter les paramètres qui ont changé
        highlight = {
            'coupon': abs(coupon_rate - prev_values['coupon_rate']) > 0.0001,
            'yield': abs(yield_rate - prev_values['yield_rate']) > 0.0001,
            'maturity': abs(maturity - prev_values['maturity']) > 0.0001
        }

        # Sauvegarder les valeurs actuelles pour la prochaine mise à jour
        self.previous_values = {
            'coupon_rate': coupon_rate,
            'yield_rate': yield_rate,
            'maturity': maturity
        }

        # Effacement des graphiques
        for ax in [self.ax_price_yield, self.ax_duration_yield, self.ax_convexity_yield,
                   self.ax_duration_maturity, self.ax_convexity_maturity, self.ax_table]:
            ax.clear()

        # Réinitialisation des titres
        self.ax_price_yield.set_title('Prix vs Rendement', fontsize=12, fontweight='bold')
        self.ax_price_yield.set_xlabel('Rendement (%)', fontsize=10)
        self.ax_price_yield.set_ylabel('Prix', fontsize=10)

        self.ax_duration_yield.set_title('Duration vs Rendement', fontsize=12, fontweight='bold')
        self.ax_duration_yield.set_xlabel('Rendement (%)', fontsize=10)
        self.ax_duration_yield.set_ylabel('Duration', fontsize=10)

        self.ax_convexity_yield.set_title('Convexité vs Rendement', fontsize=12, fontweight='bold')
        self.ax_convexity_yield.set_xlabel('Rendement (%)', fontsize=10)
        self.ax_convexity_yield.set_ylabel('Convexité', fontsize=10)

        self.ax_duration_maturity.set_title('Duration vs Maturité', fontsize=12, fontweight='bold')
        self.ax_duration_maturity.set_xlabel('Maturité (années)', fontsize=10)
        self.ax_duration_maturity.set_ylabel('Duration', fontsize=10)

        self.ax_convexity_maturity.set_title('Convexité vs Maturité', fontsize=12, fontweight='bold')
        self.ax_convexity_maturity.set_xlabel('Maturité (années)', fontsize=10)
        self.ax_convexity_maturity.set_ylabel('Convexité', fontsize=10)

        self.ax_table.set_title('Indicateurs', fontsize=12, fontweight='bold')
        self.ax_table.axis('off')

        # Grilles
        for ax in [self.ax_price_yield, self.ax_duration_yield, self.ax_convexity_yield,
                   self.ax_duration_maturity, self.ax_convexity_maturity]:
            ax.grid(True, linestyle='--', alpha=0.7)
            ax.margins(0.05)

        # Couleurs de base et couleurs de surbrillance
        base_colors = {
            'price': 'blue',
            'duration_mac': 'green',
            'duration_mod': 'orange',
            'convexity': 'purple'
        }

        highlight_colors = {
            'price': 'darkblue',
            'duration_mac': 'darkgreen',
            'duration_mod': 'darkorange',
            'convexity': 'darkviolet'
        }

        # Largeurs de ligne de base et accentuées
        base_lw = 1.8
        highlight_lw = 3.0

        # Tracé des courbes avec effets de surbrillance
        # Prix vs Yield
        prices = [calculer_prix_obligation(coupon_rate, y, maturity, self.face_value)
                  for y in self.yield_range]

        # Effet de surbrillance si le taux de coupon ou le rendement a changé
        price_lw = highlight_lw if highlight['coupon'] or highlight['yield'] else base_lw
        price_color = highlight_colors['price'] if highlight['coupon'] or highlight['yield'] else base_colors['price']

        line_price, = self.ax_price_yield.plot(self.yield_range * 100, prices,
                                               color=price_color, linewidth=price_lw)
        self.ax_price_yield.axvline(x=yield_rate * 100, color='red', linestyle='--', alpha=0.7)

        # Duration vs Yield
        durations_mac = [calculer_duration_macaulay(coupon_rate, y, maturity, self.face_value)
                         for y in self.yield_range]
        durations_mod = [calculer_duration_modifiee(coupon_rate, y, maturity, self.face_value)
                         for y in self.yield_range]

        # Effet de surbrillance si le taux de coupon ou le rendement a changé
        duration_lw = highlight_lw if highlight['coupon'] or highlight['yield'] else base_lw
        duration_color_mac = highlight_colors['duration_mac'] if highlight['coupon'] or highlight['yield'] else \
        base_colors['duration_mac']
        duration_color_mod = highlight_colors['duration_mod'] if highlight['coupon'] or highlight['yield'] else \
        base_colors['duration_mod']

        line_duration_mac, = self.ax_duration_yield.plot(self.yield_range * 100, durations_mac,
                                                         label='Macaulay', color=duration_color_mac,
                                                         linewidth=duration_lw)
        line_duration_mod, = self.ax_duration_yield.plot(self.yield_range * 100, durations_mod,
                                                         label='Modifiée', color=duration_color_mod,
                                                         linewidth=duration_lw)
        self.ax_duration_yield.axvline(x=yield_rate * 100, color='red', linestyle='--', alpha=0.7)
        self.ax_duration_yield.legend(loc='best', fontsize=9)

        # Convexité vs Yield
        convexities = [calculer_convexite(coupon_rate, y, maturity, self.face_value)
                       for y in self.yield_range]

        # Effet de surbrillance si le taux de coupon ou le rendement a changé
        convexity_lw = highlight_lw if highlight['coupon'] or highlight['yield'] else base_lw
        convexity_color = highlight_colors['convexity'] if highlight['coupon'] or highlight['yield'] else base_colors[
            'convexity']

        line_convexity, = self.ax_convexity_yield.plot(self.yield_range * 100, convexities,
                                                       color=convexity_color, linewidth=convexity_lw)
        self.ax_convexity_yield.axvline(x=yield_rate * 100, color='red', linestyle='--', alpha=0.7)

        # Duration vs Maturité
        durations_mac_mat = [calculer_duration_macaulay(coupon_rate, yield_rate, m, self.face_value)
                             for m in self.maturity_range]
        durations_mod_mat = [calculer_duration_modifiee(coupon_rate, yield_rate, m, self.face_value)
                             for m in self.maturity_range]

        # Effet de surbrillance si la maturité a changé
        duration_mat_lw = highlight_lw if highlight['maturity'] else base_lw
        duration_mat_color_mac = highlight_colors['duration_mac'] if highlight['maturity'] else base_colors[
            'duration_mac']
        duration_mat_color_mod = highlight_colors['duration_mod'] if highlight['maturity'] else base_colors[
            'duration_mod']

        line_duration_mac_mat, = self.ax_duration_maturity.plot(self.maturity_range, durations_mac_mat,
                                                                label='Macaulay', color=duration_mat_color_mac,
                                                                linewidth=duration_mat_lw)
        line_duration_mod_mat, = self.ax_duration_maturity.plot(self.maturity_range, durations_mod_mat,
                                                                label='Modifiée', color=duration_mat_color_mod,
                                                                linewidth=duration_mat_lw)
        self.ax_duration_maturity.axvline(x=maturity, color='red', linestyle='--', alpha=0.7)
        self.ax_duration_maturity.legend(loc='best', fontsize=9)

        # Convexité vs Maturité
        convexities_mat = [calculer_convexite(coupon_rate, yield_rate, m, self.face_value)
                           for m in self.maturity_range]

        # Effet de surbrillance si la maturité a changé
        convexity_mat_lw = highlight_lw if highlight['maturity'] else base_lw
        convexity_mat_color = highlight_colors['convexity'] if highlight['maturity'] else base_colors['convexity']

        line_convexity_mat, = self.ax_convexity_maturity.plot(self.maturity_range, convexities_mat,
                                                              color=convexity_mat_color,
                                                              linewidth=convexity_mat_lw)
        self.ax_convexity_maturity.axvline(x=maturity, color='red', linestyle='--', alpha=0.7)

        # TABLEAU DES INDICATEURS
        price = calculer_prix_obligation(coupon_rate, yield_rate, maturity, self.face_value)
        duration_mac = calculer_duration_macaulay(coupon_rate, yield_rate, maturity, self.face_value)
        duration_mod = calculer_duration_modifiee(coupon_rate, yield_rate, maturity, self.face_value)
        convexity = calculer_convexite(coupon_rate, yield_rate, maturity, self.face_value)

        # Création des données du tableau
        table_data = [
            ['Paramètres', 'Valeurs'],
            ['Taux coupon', f'{coupon_rate * 100:.1f}%'],
            ['Rendement', f'{yield_rate * 100:.1f}%'],
            ['Maturité', f'{maturity} ans'],
            ['Nominal', f'{self.face_value:,.0f}'],
            ['', ''],  # Ligne de séparation
            ['Indicateurs', 'Valeurs'],
            ['Prix', f'{price:,.2f}'],
            ['Duration (Mac)', f'{duration_mac:.4f}'],
            ['Duration (Mod)', f'{duration_mod:.4f}'],
            ['Convexité', f'{convexity:.4f}']
        ]

        # Création du tableau avec plus d'espace
        self.ax_table.clear()
        table = self.ax_table.table(
            cellText=table_data,
            loc='center',
            cellLoc='center',  # Centrer le texte dans les cellules
            colWidths=[0.5, 0.5],  # Réduire légèrement la largeur des colonnes
            bbox=[0.02, 0.05, 1.0, 0.9]  # [left, bottom, width, height] - Ajuster pour plus d'espace
        )

        # Style du tableau avec plus d'espacement
        table.auto_set_font_size(False)
        table.set_fontsize(10)

        # Application des styles aux cellules avec plus d'espacement
        for i in range(len(table_data)):
            for j in range(len(table_data[0])):
                cell = table[i, j]

                # Style des en-têtes
                if i == 0 or i == 6:
                    cell.set_facecolor('#e1f5fe')
                    cell.set_text_props(weight='bold')
                    cell.set_height(0.11)  # Cellules d'en-tête plus hautes
                # Ligne de séparation
                elif i == 5:
                    cell.set_facecolor('white')
                    cell.set_height(0.08)  # Espace de séparation
                # Lignes normales
                else:
                    cell.set_facecolor('#f8f9fa')
                    cell.set_height(0.110)  # Cellules normales plus hautes

                # Bordures et alignement
                cell.set_edgecolor('#cccccc')
                cell.set_linewidth(1)

                # Padding interne des cellules
                cell.PAD = 0.08  # Augmenter l'espace interne des cellules

                # Bordures et alignement
                cell.set_edgecolor('#cccccc')
                cell.set_linewidth(1)

        # Désactiver les axes
        self.ax_table.axis('off')

        # Titre du tableau
        self.ax_table.set_title('Indicateurs', fontsize=12, fontweight='bold', pad=10)

        # Ajouter des tooltips pour les courbes
        cursor = mplcursors.cursor([line_price, line_duration_mac, line_duration_mod,
                                    line_convexity, line_duration_mac_mat,
                                    line_duration_mod_mat, line_convexity_mat],
                                   hover=True)

        @cursor.connect("add")
        def on_add(sel):
            x, y = sel.target
            # Déterminer le type de graphique en fonction de la ligne sélectionnée
            if sel.artist == line_price:
                sel.annotation.set_text(f"Rendement: {x:.2f}%\nPrix: {y:.2f}")
            elif sel.artist in [line_duration_mac, line_duration_mac_mat]:
                if sel.artist == line_duration_mac:
                    sel.annotation.set_text(f"Rendement: {x:.2f}%\nDuration (Mac): {y:.4f}")
                else:
                    sel.annotation.set_text(f"Maturité: {x:.0f} ans\nDuration (Mac): {y:.4f}")
            elif sel.artist in [line_duration_mod, line_duration_mod_mat]:
                if sel.artist == line_duration_mod:
                    sel.annotation.set_text(f"Rendement: {x:.2f}%\nDuration (Mod): {y:.4f}")
                else:
                    sel.annotation.set_text(f"Maturité: {x:.0f} ans\nDuration (Mod): {y:.4f}")
            elif sel.artist in [line_convexity, line_convexity_mat]:
                if sel.artist == line_convexity:
                    sel.annotation.set_text(f"Rendement: {x:.2f}%\nConvexité: {y:.4f}")
                else:
                    sel.annotation.set_text(f"Maturité: {x:.0f} ans\nConvexité: {y:.4f}")

            # Améliorer le style de l'annotation
            sel.annotation.get_bbox_patch().set(boxstyle="round,pad=0.5",
                                                fc="#ffffffee", ec="#555555", lw=1.5)

        # Ajuster l'espacement
        # self.fig.tight_layout(rect=[0, 0, 1, 0.95])
        plt.subplots_adjust(wspace=0.175, hspace=0.66, top=0.84, right=0.918, bottom=0.036, left=0.112)
        self.ax_coupon_slider.set_position([0.1, 0.1, 0.2, 0.06])
        self.ax_yield_slider.set_position([0.42, 0.1, 0.2, 0.06])
        self.ax_maturity_slider.set_position([0.75, 0.1, 0.2, 0.06])

        self.fig.canvas.draw_idle()

    def show(self):
        """Affiche l'animation interactive"""
        plt.show()


# Point d'entrée principal
if __name__ == "__main__":
    print("Animation interactive pour la formation sur les risques liés à l'investissement obligataire")
    print("Ajustez les paramètres avec les curseurs pour observer les changements")
    try:
        animation = ObligationAnimation()
        animation.show()
    except Exception as e:
        print(f"Erreur lors de l'exécution: {e}")
        import traceback

        traceback.print_exc()
import mplcursors
import matplotlib.patches as patches
import mplcursors
